/**
 * \file sdc_session.h
 *
 * \brief libSDC: Definition of functions etc. for session management - <b>should not be included directly!</b>
 *
 * Application should not include this header directly!
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

/**
 * \ingroup libSDC_API
 * \defgroup sessionmanagement Session management
 * Functions to manage sessions
 */

#ifndef __SDC_LIB_SESSION_H_
#define __SDC_LIB_SESSION_H_

#ifdef __cplusplus
extern "C"
{
#endif

#include <sdc.h>
#include <sdc_perm.h>

/* Definitions types and defaults */

/* Functions */

/**
 * \ingroup sessionmanagement
 * \brief Create a SDC session
 *
 * Creates a new session and allocates the \ref sdc_session_t.
 * \ref sdc_session_t is needed internally to keep track of the session.
 * After usage it needs to be freed using ::sdc_close_session
 *
 * \param[out] session   pointer to return the session handle.
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given pointer to session is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory for session struct
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_open_session(sdc_session_t **session);


/* Note: validity of session needs to be checked internally */
/**
 * \ingroup sessionmanagement
 * \brief Close a SDC session
 *
 * Close and free a SDC session and all associated resources.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_close_session(sdc_session_t *session);

/* Note: validity of session needs to be checked internally */
/**
 * \ingroup sessionmanagement
 * \brief Assign the key storage key with the given ID to the session.
 *
 * This function is used to assign a key storage key with ID \p kid to the session.
 * All cryptographic operations (see \ref operations) will use this key until the
 * key is unset or replaced by a different key.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] kid    ID of the key storage key
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_KEY_INVALID - given KID is invalid
 * \return \ref SDC_KID_NOT_AVAILABLE - the given KID is valid but not available in the key storage
 * \return \ref SDC_ACCESS_DENIED - the process has no permission to perform any operation using the given key
 * \return \ref SDC_KEY_NOT_READY - key not ready - try again later
 * \return \ref SDC_KEY_LOCKED - key store currently locked - try again later
 * \return \ref SDC_KEY_TAMPERED - loading the key aborted as it seems to be tampered
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_session_load_storage_key(sdc_session_t *session, sdc_key_id_t kid);

/* Note: validity of session needs to be checked internally */
/**
 * \ingroup sessionmanagement
 * \brief Assign the built in key to the session.
 *
 * This function is used to generate and assign a built key to the session.
 * All cryptographic operations (see \ref operations) will use this key until the
 * key is unset or replaced by a different key.
 *
 * In case the user Id in \p perms is \ref SDC_FLAG_INVALID_UID the user has no access rights at all.
 * In case the group ID in \p perms is \ref SDC_FLAG_INVALID_GID the group has no access rights at all.
 *
 * \param[in] session          session handle created using ::sdc_open_session
 * \param[in] fmt              format of the key
 * \param[in] len              length of the key
 * \param[in] modifier         buffer containing modifier data (key identifier)
 * \param[in] modifier_len     length of the modifier data
 * \param[in] secret_modifier  does the modifier contain secret data
 * \param[in] perms            Access permissions exactly as specified when using this key
 *                             for the first time (used during key derivation).
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_KEY_FMT_INVALID - \p fmt is no valid member of \ref sdc_key_fmt_t
 * \return \ref SDC_KEYLEN_INVALID - \p len is no valid member of \ref sdc_key_len_t
 * \return \ref SDC_ACCESS_DENIED - the process has no permission to perform any operation using the given key
 * \return \ref SDC_INVALID_PARAMETER - no modifier passed, but modifier_len not zero
 * \return \ref SDC_PERM_INVALID - a permission exceeds the max allowed range
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_session_load_builtin_key(sdc_session_t *session,
                                                 const sdc_key_fmt_t fmt,
                                                 const sdc_key_len_t len,
                                                 const uint8_t *modifier,
                                                 const size_t modifier_len,
                                                 const bool secret_modifier,
                                                 const sdc_permissions_t *perms);

/**
 * \ingroup keymanagement_keystore
 * \brief Try to load the key described in the formatted memory buffer (with secret modifier)
 *
 * Load the key described in the formatted memory buffer.\n
 *
 * \p secret_mod_data must only provided if a key with secret modifier has been used when generating
 * the formatted data.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] formatted_data   pointer to the buffer containing the formatted data to be unwrapped (externally provided buffer)
 * \param[in] formatted_len    length of the \p formatted_data buffer (external specified length)
 * \param[in] secret_mod_data  pointer to the secret modifier (NULL if no modifier)
 * \param[in] secret_mod_len   length of the \p secret_mod_data
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_NOT_SUPPORTED - Key kind is not supported
 * \return \ref SDC_KEY_INVALID - given KID is invalid
 * \return \ref SDC_KID_NOT_AVAILABLE - the given KID is valid but not available in the key storage
 * \return \ref SDC_ACCESS_DENIED - the process has no permission to perform any operation using the given key
 * \return \ref SDC_KEY_NOT_READY - key not ready - try again later
 * \return \ref SDC_KEY_LOCKED - key store currently locked - try again later
 * \return \ref SDC_KEY_TAMPERED - loading the key aborted as it seems to be tampered
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_AUTOLOAD_KEY_UNSUPPORTED - the key type of the formatted data is not supported
 * \return \ref SDC_AUTOLOAD_KEY_WITH_SECRET - the key of the formatted data requires a secret modifier to be specified
 * \return \ref SDC_MODIFIER_INVALID - the secret modifier is invalid for the key or its length is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_import_formatted_autoload_key_with_secret_mod(
    sdc_session_t *session,
    const uint8_t *formatted_data, const size_t formatted_len,
    const uint8_t *secret_mod_data, const size_t secret_mod_len);
/**
 * \ingroup sessionmanagement
 * \brief Remove assigned key from session
 *
 * This function is used to unset the assigned key of a session.
 * All cryptographic operations (which require a key) will fail afterwards.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_KEY_UNSET - no key set in session
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_session_unset_key(sdc_session_t *session);


#ifdef __cplusplus
}
#endif

#endif
